/***************************************************************************
 *
 * Copyright (c) 2013,2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <iostream>
#include <cmath>
#include <limits>

#include "Log.h"
#include "CalibrationHelpers.h"
#include "Calibration.h"
#include "Matrix.h"

using namespace std;
using namespace LayerManagerCalibration;

bool Calibration::parseFile(const string& fileName, bool store_param)
{
    ifstream stream(fileName.c_str());
    string line;
    float x[6];
    bool valid = true;
    unsigned char loop = 0;
    char current_byte;
    char flag = false;

    char *coefficients_array[6]	=	{
                                                    "x coefficient one",
                                                    "x coefficient two",
                                                    "x coefficient three",
                                                    "y coefficient one",
                                                    "y coefficient two",
                                                    "y coefficient three" };
    if (!stream.good())
    {
        LOG_WARNING("Calibration",
                    "Calibration file, name=" << fileName
                    << " not found or not readable");
        m_isValid = false;
        return false;
    }

    while (getline(stream, line) && (loop < 6))
    {
        size_t delimiter = line.find_first_of(':');
        if (delimiter == string::npos)
        {
            LOG_WARNING("Calibration",
                        "Line found without colon, line=" << line);
            return false;
        }

        string key = line.substr(0, delimiter);
        string value = line.substr(delimiter + 1);

        key = trim(key);
        value = trim(value);

        if (key.compare(coefficients_array[loop]) == 0)
        {
            istringstream iss(value);
            iss >> x[loop];
            iss.clear();
            while(iss)
            {
                    current_byte = iss.peek();

                    if( std::isalpha(current_byte) ||\
                                        ( flag && ( current_byte == '.' ) ) )
                    {
                            valid = false;
                            break;
                    }

                    if( current_byte == '.' )
                    {
                        flag = true;
                    }
            }

            if(true == valid)
            {
                LOG_INFO("Calibration",
                            "Found valid "<< coefficients_array[loop]<< ", value=" << value);
            }
            else
            {
                LOG_WARNING("Calibration","Found invalid  "<< coefficients_array[loop]<< ", "
                        "value=" << value <<", x="<<x[loop]);
                    break;
            }
    }
    loop++;
    }

    if (valid)
    {
        if (store_param)
        {
            m_fXCoefficientOne = x[0];
            m_fXCoefficientTwo = x[1];
            m_fXCoefficientThree = x[2];
            m_fYCoefficientOne = x[3];
            m_fYCoefficientTwo = x[4];
            m_fYCoefficientThree = x[5];
        }
        m_isValid = true;
    }
    else
    {
        m_isValid = false;
        return false;
    }

    return true;
}

bool Calibration::write(ostream& os)
{
    if (!os.good())
    {
        LOG_WARNING("Calibration", "Output stream not valid");
        return false;
    }

    os << "x coefficient one: " << m_fXCoefficientOne << "\n"
       << "x coefficient two: " << m_fXCoefficientTwo << "\n"
       << "x coefficient three: " << m_fXCoefficientThree << "\n"
       << "y coefficient one: " << m_fYCoefficientOne << "\n"
       << "y coefficient two: " << m_fYCoefficientTwo << "\n"
       << "y coefficient three: " << m_fYCoefficientThree << endl;

    if (os.fail())
    {
        LOG_WARNING("Calibration",
                    "Failed to write configuration");
        return false;
    }
    else
    {
        return true;
    }
}

void Calibration::transformCoordinate(const coordinate& coordIn,
                                      coordinate& coordOut)
{
    int xval = m_fXCoefficientOne * coordIn.x
             + m_fXCoefficientTwo * coordIn.y
             + m_fXCoefficientThree;
    int yval = m_fYCoefficientOne * coordIn.x
             + m_fYCoefficientTwo * coordIn.y
             + m_fYCoefficientThree;

    if (xval < 0)
    {
        coordOut.x = 0;
    }
    else
    {
        coordOut.x = xval;
    }

    if (yval < 0)
    {
        coordOut.y = 0;
    }
    else
    {
        coordOut.y = yval;
    }
}

bool Calibration::inverseTransformCoordinate(const coordinate& coordIn, 
                                             coordinate& coordOut)
{
    double x, y;
    double denominator = ((m_fXCoefficientOne * m_fYCoefficientTwo)
                           - (m_fXCoefficientTwo * m_fYCoefficientOne));

    // Check for zero denominator cases
    if (fabs(denominator) < numeric_limits<double>::epsilon())
    {
        LOG_WARNING("Calibration",
                    "Unable to calculate inverse transformation");

        return false;
    }
    else
    {
        y = ( (  m_fXCoefficientOne * (double)coordIn.y   )
              - (m_fYCoefficientOne * (double)coordIn.x   )
              - (m_fXCoefficientOne * m_fYCoefficientThree)
              + (m_fYCoefficientOne * m_fXCoefficientThree) )
            / denominator;

        x = ( (  m_fYCoefficientTwo * (double)coordIn.x   )
              - (m_fXCoefficientTwo * (double)coordIn.y   )
              + (m_fXCoefficientTwo * m_fYCoefficientThree)
              - (m_fYCoefficientTwo * m_fXCoefficientThree) )
            / denominator;
    }

    coordOut.x = (x > 0) ? floor(x): 0;

    coordOut.y = (y > 0) ? floor(y): 0;

    return true;
}

bool Calibration::calculateCoefficients(const coordinate& measured1,
                                        const coordinate& measured2,
                                        const coordinate& measured3,
                                        const coordinate& expected1,
                                        const coordinate& expected2,
                                        const coordinate& expected3,
                                        float& xCoefficientOne,
                                        float& xCoefficientTwo,
                                        float& xCoefficientThree,
                                        float& yCoefficientOne,
                                        float& yCoefficientTwo,
                                        float& yCoefficientThree)
{
    // Please see slyt277.pdf - "Calibration in touch-screen systems" for
    // further information.
    // http://www.ti.com/lit/an/slyt277/slyt277.pdf

    Matrix::Data init = {{(double)measured1.x, (double)measured1.y, 1},
                         {(double)measured2.x, (double)measured2.y, 1},
                         {(double)measured3.x, (double)measured3.y, 1}};
    bool result = false;

    Matrix A(init);

    double determinant = A.calculateDeterminant();

    if (fabs(determinant - 0) < numeric_limits<double>::epsilon())
    {
        LOG_WARNING("Calibration",
                    "Unable to calculate coefficients");

        result = false;
    }
    else
    {
        // Invert the matrix
        A.adjugateMatrix();
        A = A * (1/determinant);

        vector3 xCoordinates = {{ (double)expected1.x,
                                 (double)expected2.x,
                                 (double)expected3.x }};

        vector3 xResult = A * xCoordinates;

        vector3 yCoordinates = {{ (double)expected1.y,
                                 (double)expected2.y,
                                 (double)expected3.y }};

        vector3 yResult = A * yCoordinates;

        xCoefficientOne   = xResult.value[0];
        xCoefficientTwo   = xResult.value[1];
        xCoefficientThree = xResult.value[2];
        yCoefficientOne   = yResult.value[0];
        yCoefficientTwo   = yResult.value[1];
        yCoefficientThree = yResult.value[2];

        result = true;
    }

    return result;
}
